DOUBLE PRECISION FUNCTION norma_vetor(array, n)
    !Calcula a norma de um dado vetor, com dada dimensao n

    IMPLICIT NONE

    INTEGER :: n
    DOUBLE PRECISION, DIMENSION(n), INTENT(IN) :: array

    !Cálculo da norma
    norma_vetor = SQRT(SUM(array * array))

END FUNCTION norma_vetor

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

SUBROUTINE eliminacao(matriz, n, array_b)
    !Recebe uma matriz de coeficientes nxn, a triangulariza por pivotamento parcial e realiza as mesmas operacoes no vetor array_b de termos independentes.
    
    IMPLICIT NONE
    
    DOUBLE PRECISION, DIMENSION(n,n) :: matriz !Matriz a ser triangularizada
    DOUBLE PRECISION, DIMENSION(n) :: aux_matriz, array_b !Array auxiliar nos cálculos e array de termos independentes
    DOUBLE PRECISION :: x, aux_array !Variáveis auxiliares de cálculos
    INTEGER :: n, i, j !Dimensão n da matriz e dos arrays acima; variáveis auxiliares dos cálculos
    
    !Laço que varre linhas da matriz
    DO i = 1, n

        !Laço que varre as linhas adjacentes
        j = i + 1    

        DO WHILE (j <= n)
        
            !Compara "candidatos" a pivô parcial
            IF (ABS(matriz(i,i)) < ABS(matriz(j,i))) THEN
            
                !Troca as linhas colocando na posição correta a qual contém o maior pivô e troca as mesmas linhas no array_b
                aux_matriz(i:n) = matriz(i,i:n)
                matriz(i,i:n) = matriz(j,i:n)
                matriz(j,i:n) = aux_matriz(i:n)
                
                aux_array = array_b(i)
                array_b(i) = array_b(j)
                array_b(j) = aux_array

            END IF

            j = j + 1

        END DO

        !Laço que faz a eliminação
        j = i + 1

        DO WHILE (j <= n)
        
            !Define o multiplicador x do pivô, faz a eliminação e aplica o multiplicador no array_b
            x = matriz(j,i) / matriz(i,i)
            matriz(j,i:n) = matriz(i,i:n) * x - matriz(j,i:n)
            array_b(j) = array_b(i) * x - array_b(j)
            
            j = j + 1

        END DO
    
    END DO

END SUBROUTINE eliminacao

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

SUBROUTINE substituicao(matriz, array_x, array_b, n)
    !Recebe uma matriz nxn e dois arrays de dimensão n. A partir da matriz e array_b, encontra a solução array_x de um sistema linear pelo método de Gauss (já feita a etapa de eliminação)

    IMPLICIT NONE

    DOUBLE PRECISION, DIMENSION(n,n) :: matriz !Matriz já triangularizada
    DOUBLE PRECISION, DIMENSION(n), INTENT(OUT) :: array_x !Array solução do sistema
    DOUBLE PRECISION, DIMENSION(n) :: array_b !Array b de termos independentes, modificado pela triangularizacao de A
    DOUBLE PRECISION :: soma, aux !Variáveis auxiliares dos cálculos
    INTEGER :: n, i, j !Dimensão n dos arrays/matriz acima e variáveis auxiliares

    !Determina os elementos do array_x por back substitution
    array_x(n) = array_b(n) / matriz(n,n)
    
    DO i = n - 1, 1, -1

        !Laço que determina a somatoria no cálculo dos elementos array_x(i)
        j = i + 1
        soma = 0.

        DO WHILE (j <= n)

            soma = soma + matriz(i,j) * array_x(j)

            j = j + 1

        END DO

        !Cálculo propriamente dito
        array_x(i) = (array_b(i) - soma) / matriz(i,i)

    END DO

END SUBROUTINE substituicao

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

SUBROUTINE gauss(matrizA, vetorX, vetorB, n, condicionamento)
    !Resolvendo um sistema linear pelo método de eliminação de Gauss

    IMPLICIT NONE

    DOUBLE PRECISION, DIMENSION(n,n) :: matrizA, matrizA_norm !Matriz de coeficientes original e normalizada
    DOUBLE PRECISION, DIMENSION(n) :: vetorX, vetorB !Vetores de incógnitas (solução) e de termos independentes
    INTEGER :: i, j, n !Variáveis auxiliares e dimensao n dos arrays/matrizes
    DOUBLE PRECISION, EXTERNAL :: norma_vetor !Função que calcula a norma de um vetor
    DOUBLE PRECISION :: mod_det !Módulo do determinante da matriz normalizada
    LOGICAL :: condicionamento !Indica o condicionamento do sistema

    !CONDICIONAMENTO DO SISTEMA
    !Primeiro: normalização da matriz de coeficientes
    DO i = 1, n
        
        matrizA_norm(i,:) = matrizA(i,:) / norma_vetor(matrizA(i,:), n)
        vetorB(i) = vetorB(i) / norma_vetor(matrizA(i,:), n)

    END DO

    !Segundo: encontrando o determinante da matriz normalizada
    CALL eliminacao(matrizA_norm, n, vetorB)
    mod_det = 1.

    DO i = 1, n

        mod_det = mod_det * matrizA_norm(i,i)

    END DO

    mod_det = ABS(mod_det)

    !Verificando condicionamento e, caso bem condicionado, resolve o sistema
    condicionamento = .FALSE.

    IF (mod_det > 1.E-3) THEN
        condicionamento = .TRUE.

        CALL substituicao(matrizA_norm, vetorX, vetorB, n)

        PRINT *, "A solução encontrada para o sistema foi:"
        PRINT *, vetorX

    ELSE IF (mod_det == 0.) THEN

        PRINT *, "O sistema é indeterminado."

    ELSE

        PRINT *, "O sistema é mal condicionado."

    END IF
    
END SUBROUTINE gauss

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

SUBROUTINE circulo(pontos, n, m)
    !Determina a coordenada (a, b) do centro de um círculo de raio r e k = a^2 + b^2 - r^2 que passa por 3 pontos (x, y) dados

    IMPLICIT NONE

    DOUBLE PRECISION, DIMENSION(n, m) :: pontos !Matriz que contém os pontos
    DOUBLE PRECISION, DIMENSION(n, n) :: matrizA !Matriz de coeficientes
    DOUBLE PRECISION, DIMENSION(n) :: vetorB, vetorX !Vetor de termos independentes e vetor solução (a, b, k)
    INTEGER :: i, n, m !Variável de auxílio aos cálculos e dimensões das matrizes e vetores
    DOUBLE PRECISION :: r !Raio do círculo r = raiz(a^2 + b^2 - k)
    LOGICAL :: condicionamento !Se condicionamento for True, retorna os parâmetros a, b e r

    !Construção da matriz de coeficientes e do vetor de termos independentes
    DO i = 1, n

        matrizA(i,1:m) = 2 * pontos(i,1:m)
        matrizA(i, n) = -1

        vetorB(i) = pontos(i, 1) * pontos(i, 1) + pontos(i, 2) * pontos(i, 2) !Isso só vale para m = 2

    END DO

    !Resolve o sistema por eliminação de Gauss
    CALL gauss(matrizA, vetorx, vetorB, n, condicionamento)

    !Retorno caso o sistema seja determinado e bem condicionado
    IF (condicionamento .EQV. .TRUE.) THEN

        PRINT *, "O centro do círculo determinado pelos pontos dados se encontra na posição (a, b) =", vetorX(1:2)

        r = SQRT(vetorX(1) * vetorX(1) + vetorX(2) * vetorX(2) - vetorX(3))
        PRINT *, "E o raio do círculo é", r

    END IF

END SUBROUTINE circulo

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

PROGRAM circulos
    !Recebe um conjunto de 3 pontos e retorna as coordenadas do centro do círculo formado por eles, como também o raio do mesmo.
    !Caso os pontos estejam muitos próximos ou sobre uma reta, retorna um aviso.

    DOUBLE PRECISION, DIMENSION(3,2) :: pontos, pontos_malcond, pontos_indet !Pontos fornecidos
    
    pontos = RESHAPE((/1., 1., 2., 1., 2., 2./), (/3,2/)) !Sistema determinado e bem condicionado
    pontos_malcond = RESHAPE((/1., 4., 7., -2., 0.01, 2./), (/3,2/)) !Sistema mal condicionado
    pontos_indet = RESHAPE((/1., 1.5, 2., 1., 1., 1./), (/3,2/)) !Sistema indeterminado

    PRINT *,

    !Saída do programa    
    PRINT *, "PONTOS: (1, 1) (1, 2) (2, 2)"
    CALL circulo(pontos, 3, 2) !Resolve o sistema
    PRINT *,
    PRINT *, "--------------------------------------------------"
    PRINT *,

    PRINT *, "PONTOS: (1, -2) (4, 0.01) (7, 2)"
    CALL circulo(pontos_malcond, 3, 2) !Resolve o sistema
    PRINT *,
    PRINT *, "--------------------------------------------------"
    PRINT *,

    PRINT *, "PONTOS: (1, 1) (1.5, 1) (2, 1)"
    CALL circulo(pontos_indet, 3, 2) !Resolve o sistema
    PRINT *,
    PRINT *, "--------------------------------------------------"
    PRINT *,

END PROGRAM circulos
